//////////////////////////////////////////////////////////
// CPU routines

#include <string.h>

#include "vb_io.h"
#include "v810_opt.h"
#include "v810_cpu.h"
#include "v810_cpuD.h"
#include "vb_vbt.h"
#include "vb_vbtD.h"
#include "vb_dsp.h"
//Options...
#include "vb_set.h"

////////////////////////////////////////////////////////////
// Globals
WORD P_REG[32];  // Program registers pr0-pr31
WORD S_REG[32];  // System registers sr0-sr31
WORD PC;         // Program Counter

const BYTE opcycle[0x50] = {
	0x01,0x01,0x01,0x01,0x01,0x01,0x03,0x01,0x0D,0x26,0x0D,0x24,0x01,0x01,0x01,0x01,
	0x01,0x01,0x01,0x01,0x01,0x01,0x03,0x01,0x0F,0x0A,0x05,0x00,0x01,0x01,0x03,0x00, //CLI, HALT, LDSR, STSR, SEI, BSTR -- Unknown clocks
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0x03,0x03,0x01,0x01,0x01,0x01,
	0x01,0x01,0x0D,0x01,0x01,0x01,0x00,0x01,0x03,0x03,0x1A,0x05,0x01,0x01,0x00,0x01, //these are based on 16-bit bus!! (should be 32-bit?)
	0x01,0x01,0x01,0x01,0x01,0x03,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01
};


//////////////////////////////////////////////////////

// Reinitialize the defaults in the CPU
void v810_reset() {
    P_REG[0]      =  0x00000000;
    PC            =  0xFFFFFFF0;
    S_REG[ECR]    =  0x0000FFF0;
    S_REG[PSW]    =  0x00008000;
    S_REG[PIR]    =  0x00005346;
    S_REG[TKCW]   =  0x000000E0;
}

//could be done a lot better, later maby!
int v810_trc(int dbg_trc) {
    int lowB, highB, lowB2, highB2;             // up to 4 bytes for instruction (either 16 or 32 bits)
    static int opcode;
	int arg1 = 0;
	int arg2 = 0;
	int arg3 = 0;
    int tmp2;
    int j;
    int flags = 0;
    INT64 temp = 0;
    INT64U tempu = 0;
    int val = 0;
    WORD msb = 0;
    int tCount = 0;
	static unsigned int clocks;
	static int lastop,lastclock;

	for(;;) {

		serviceInt(clocks);
		if (serviceDisplayInt(clocks)) return 0; //serviceDisplayInt() returns with 1 when the screen needs to be redrawn

        PC = (PC&0x07FFFFFE);

#ifdef DBG_PRINT
        // Interactive Dissasemble, remove in release...
        if(tVBOpt.DISASM) { //turn on and off
            v810_addDasm(PC);
        }

        //Special Stack Trace (of sourts), remove in release
        vb_addQueue(PC); // Circular queue of last 100 instructions
#endif
		if ((PC>>24) == 0x05) { //RAM
			PC     = (PC & V810_VB_RAM.highaddr);
			lowB   = ((BYTE *)(V810_VB_RAM.off + PC))[0];
			highB  = ((BYTE *)(V810_VB_RAM.off + PC))[1];
			lowB2  = ((BYTE *)(V810_VB_RAM.off + PC))[2];
			highB2 = ((BYTE *)(V810_VB_RAM.off + PC))[3];
		}
		else if ((PC>>24) >= 0x07) { //ROM
			PC     = (PC & V810_ROM1.highaddr);
			lowB   = ((BYTE *)(V810_ROM1.off + PC))[0];
			highB  = ((BYTE *)(V810_ROM1.off + PC))[1];
			lowB2  = ((BYTE *)(V810_ROM1.off + PC))[2];
			highB2 = ((BYTE *)(V810_ROM1.off + PC))[3];
		}
		else {
			dtprintf(10,ferr,"\nInvalid PC - %08X",PC);
#ifdef DBG_PRINT
			vb_dumpQueue();
#endif
			return 1;
		}

        P_REG[0]=0; //Zero the Zero Reg!!!

		tmp2 = ((PC&0x00FFFFFF)>>1);

		if ((opcode >0) && (opcode < 0x50)) { //hooray for instruction cache! (cache only if last opcode was not bad!)
			lastop = opcode;
			lastclock = opcycle[opcode];
		}

        opcode = highB >> 2;
        if((highB & 0xE0) == 0x80)        // Special opcode format for          
            opcode = (highB >> 1);    // type III instructions.

        if((opcode > 0x4F) || (opcode < 0)) {
            //Error Invalid opcode!
            dtprintf(10,ferr,"\n%08lx\t\t%2x %2x  ;Invalid Opcode", PC, lowB, highB);
			return 1;
        }

        clocks += opcycle[opcode];
		//decode opcode and arguments form packed instruction
        switch(addr_mode[opcode]) {
          case AM_I:       // Do the same Ither way =)
          case AM_II:
            arg1 = (lowB & 0x1F);
            arg2 = (lowB >> 5) + ((highB & 0x3) << 3);
            PC += 2;   // 16 bit instruction
            break;

          case AM_III:
            arg1 = ((highB & 0x1) << 8) + (lowB & 0xFE);
            break;

          case AM_IV:
            arg1 = ((highB & 0x3) << 24) + (lowB << 16) + (highB2 << 8) + lowB2;
            break;

          case AM_V:       
            arg3 = (lowB >> 5) + ((highB & 0x3) << 3);
            arg2 = (lowB & 0x1F);
            arg1 = (highB2 << 8) + lowB2;
            PC += 4;   // 32 bit instruction
            break;

          case AM_VIa:  // Mode6 form1
            arg1 = (highB2 << 8) + lowB2;
            arg2 = (lowB & 0x1F);
            arg3 = (lowB >> 5) + ((highB & 0x3) << 3);
            PC += 4;   // 32 bit instruction
            break;

          case AM_VIb:  // Mode6 form2
            arg1 = (lowB >> 5) + ((highB & 0x3) << 3);
            arg2 = (highB2 << 8) + lowB2;                              //  whats the order??? 2,3,1 or 1,3,2
            arg3 = (lowB & 0x1F);
            PC += 4;   // 32 bit instruction
            break;

          case AM_VII:   // Unhandled
            dtprintf(6,ferr,"\n%08lx\t\t%2x %2x %2x %2x", PC, lowB, highB, lowB2, highB2);
            PC +=4; // 32 bit instruction
            break;

          case AM_VIII:  // Unhandled
            dtprintf(6,ferr,"\n%08lx\t%\t2x %2x %2x %2x", PC, lowB, highB, lowB2, highB2);
            PC += 4;   // 32 bit instruction
            break;

          case AM_IX:
            arg1 = (lowB & 0x1); // Mode ID, Ignore for now
            PC += 2;   // 16 bit instruction
            break;

          case AM_BSTR:  // Bit String Subopcodes
            arg1 = (lowB >> 5) + ((highB & 0x3) << 3);
            arg2 = (lowB & 0x1F);
            PC += 2;   // 16 bit instruction
            break;

          case AM_FPP:   // Floating Point Subcode
            arg1 = (lowB >> 5) + ((highB & 0x3) << 3);
            arg2 = (lowB & 0x1F);
            arg3 = ((highB2 >> 2)&0x3F);
            PC += 4;   // 32 bit instruction
            break;

          case AM_UDEF:  // Invalid opcode.
          default:           // Invalid opcode.
            dtprintf(6,ferr,"\n%08lx\t\t%2x %2x  ;Invalid Opcode", PC, lowB, highB);
            PC += 2;                                                
            break;
        }

		//process opcode & set flags
        switch(opcode) {
          case MOV:
            P_REG[arg2] = P_REG[arg1];
            break;

          case ADD:
            flags = 0;
            temp = P_REG[arg2] + P_REG[arg1];
            // Set Flags
            if ((long)temp == 0) flags = flags | PSW_Z;
            if ((long)temp & 0x80000000)  flags = flags | PSW_S;
			if (temp < P_REG[arg2]) flags = flags | PSW_CY;
            if(((P_REG[arg2]^(~P_REG[arg1]))&(P_REG[arg2]^temp))&0x80000000) flags = flags | PSW_OV;

            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            P_REG[arg2] = (long)temp;
            break;

          case SUB:
            flags = 0;
			temp = (INT64)((INT64U)(P_REG[arg2])-(INT64U)(P_REG[arg1]));
            // Set Flags
            if ((long)temp == 0) flags = flags | PSW_Z;
            if ((long)temp & 0x80000000)  flags = flags | PSW_S;
            if(((P_REG[arg2]^P_REG[arg1])&(P_REG[arg2]^temp))&0x80000000) flags = flags | PSW_OV;
			if ((INT64U)(temp) >> 32) flags = flags | PSW_CY;

            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            P_REG[arg2] = (long)temp;
            break;

          case CMP:
            flags = 0;
			temp = (INT64)((INT64U)(P_REG[arg2])-(INT64U)(P_REG[arg1]));
            // Set Flags
            if ((long)temp == 0) flags = flags | PSW_Z;
            if ((long)temp & 0x80000000)  flags = flags | PSW_S;
            if(((P_REG[arg2]^P_REG[arg1])&(P_REG[arg2]^temp))&0x80000000)
				flags = flags | PSW_OV;
			if ((INT64U)(temp) >> 32) flags = flags | PSW_CY;
            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            break;

          case SHL:
            flags = 0;
            val = P_REG[arg1] & 0x1F;
            // set CY before we destroy the regisrer info....
            if((val != 0)&&(P_REG[arg2] >> (32 - val))&0x01) flags = flags | PSW_CY;
            P_REG[arg2] = P_REG[arg2] << val;
            // Set Flags
            if (P_REG[arg2] == 0) flags = flags | PSW_Z;
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            break;
			
          case SHR:
            flags = 0; 
            val = P_REG[arg1] & 0x1F;
            // set CY before we destroy the regisrer info....
			if ((val) && ((P_REG[arg2] >> (val-1))&0x01)) flags = flags | PSW_CY;
            P_REG[arg2] = P_REG[arg2] >> val;
            // Set Flags
            if (P_REG[arg2] == 0) flags = flags | PSW_Z;
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            break;

          case JMP:
            PC = (P_REG[arg1] & 0xFFFFFFFE);
            break;

          case SAR:
            flags = 0;
            val = P_REG[arg1] & 0x1F;
            msb = P_REG[arg2] & 0x80000000; // Grab the MSB
			
			//carry is last bit shifted out
			if( (val) && ((P_REG[arg2]>>(val-1))&0x01) )
				flags = flags | PSW_CY;

            for(j = 0; j < val; j++)
                P_REG[arg2] = (P_REG[arg2] >> 1)|msb; // Apend the MSB to the end
            
            // Set Flags
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            if (!P_REG[arg2]) flags = flags | PSW_Z;
            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            break;

          case MUL:
            flags=0;
            temp = (INT64)P_REG[arg1] * (INT64)P_REG[arg2];
            P_REG[30]   = (long)(temp >> 32);
			P_REG[arg2] = (long)temp;

            // Set Flags
			if (temp != sign_32((long)temp)) flags = flags | PSW_OV;
            if ((long)temp == 0) flags = flags | PSW_Z;
            if ((long)temp & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;

            break;

          case DIV:
            flags = 0;
            if((long)P_REG[arg1] == 0) { // Div by zero error
                // Generate exception!
				v810_exp(8, 0xFF80);
            } else {
                if((P_REG[arg2]==0x80000000)&&(P_REG[arg1]==0xFFFFFFFF)) {
					flags = flags |PSW_OV;
					P_REG[30]=0;
					P_REG[arg2] = 0x80000000;
				} else {
					temp        = (long)P_REG[arg2] % (long)P_REG[arg1];
					P_REG[arg2] = (long)P_REG[arg2] / (long)P_REG[arg1];
					if (arg2 != 30) P_REG[30] = (long)temp;
				}

				// Set Flags
				if (P_REG[arg2] == 0) flags = flags | PSW_Z;
				if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
				S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags; 
            }
            break;

          case MULU:
            flags = 0;
            tempu = (INT64U)P_REG[arg1] * (INT64U)P_REG[arg2];
            P_REG[30]   = (long)(tempu >> 32);
			P_REG[arg2] = (WORD)tempu;

            // Set Flags
			if (tempu != (long)tempu) flags = flags | PSW_OV;
            if ((long)tempu == 0) flags = flags | PSW_Z;
            if ((long)tempu & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;
            break;

          case DIVU:
            flags = 0;
            if(P_REG[arg1] == 0) { // Div by zero error
                // Generate exception!
				v810_exp(8, 0xFF80);
            } else {
				temp        = (WORD)P_REG[arg2] % (WORD)P_REG[arg1];
                P_REG[arg2] = (WORD)P_REG[arg2] / (WORD)P_REG[arg1];
				if (arg2 != 30) P_REG[30] = (long)temp;
                // Set Flags
                if (P_REG[arg2] == 0) flags = flags | PSW_Z;
                if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
                S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;
            }
            break;

          case OR:
            flags = 0;
            P_REG[arg2] = P_REG[arg1] | P_REG[arg2];
            // Set Flags
            if (P_REG[arg2] == 0) flags = flags | PSW_Z;
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;
            break;

          case AND:
            flags = 0;
            P_REG[arg2] = P_REG[arg1] & P_REG[arg2];
            // Set Flags
            if (P_REG[arg2] == 0) flags = flags | PSW_Z;
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;
            break;

          case XOR:
            flags = 0;
            P_REG[arg2] = P_REG[arg1] ^ P_REG[arg2];
            // Set Flags
            if (P_REG[arg2] == 0) flags = flags | PSW_Z;
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;
            break;

          case NOT:
            flags = 0;
			P_REG[arg2] = ~P_REG[arg1];
            // Set Flags
            if (P_REG[arg2] == 0) flags = flags | PSW_Z;
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;
            break;

          case MOV_I:
            P_REG[arg2] = sign_5(arg1);
            break;

          case ADD_I:
            flags = 0;
            temp = P_REG[arg2] + sign_5(arg1);
            // Set Flags
            if ((long)temp == 0) flags = flags | PSW_Z;
            if ((long)temp & 0x80000000)  flags = flags | PSW_S;
            if(((P_REG[arg2]^(~sign_5(arg1)))&(P_REG[arg2]^temp))&0x80000000) flags = flags | PSW_OV;
			if (temp < P_REG[arg2]) flags = flags | PSW_CY;

            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            P_REG[arg2] = (WORD)temp;
            break;

          case SETF:
			//SETF may contain bugs
			P_REG[arg2] = 0;
			switch (arg1 & 0x0F) {
				case COND_V:
					if (S_REG[PSW] & PSW_OV) P_REG[arg2] = 1;
					break;
				case COND_C:
					if (S_REG[PSW] & PSW_CY) P_REG[arg2] = 1;
					break;
				case COND_Z:
					if (S_REG[PSW] & PSW_Z) P_REG[arg2] = 1;
					break;
				case COND_NH:
					if (S_REG[PSW] & (PSW_CY|PSW_Z)) P_REG[arg2] = 1;
					break;
				case COND_S:
					if (S_REG[PSW] & PSW_S) P_REG[arg2] = 1;
					break;
				case COND_T:
					P_REG[arg2] = 1;
					break;
				case COND_LT:
					if ((!!(S_REG[PSW]&PSW_S))^(!!(S_REG[PSW]&PSW_OV))) P_REG[arg2] = 1;
					break;
				case COND_LE:
					if (((!!(S_REG[PSW]&PSW_S))^(!!(S_REG[PSW]&PSW_OV)))|(S_REG[PSW]&PSW_Z)) P_REG[arg2] = 1;
					break;
				case COND_NV:
					if (!(S_REG[PSW] & PSW_OV)) P_REG[arg2] = 1;
					break;
				case COND_NC:
					if (!(S_REG[PSW] & PSW_CY)) P_REG[arg2] = 1;
					break;
				case COND_NZ:
					if (!(S_REG[PSW] & PSW_Z)) P_REG[arg2] = 1;
					break;
				case COND_H:
					if (!(S_REG[PSW] & (PSW_CY|PSW_Z))) P_REG[arg2] = 1;
					break;
				case COND_NS:
					if (!(S_REG[PSW] & PSW_S)) P_REG[arg2] = 1;
					break;
				case COND_F:
					//always false! do nothing more
					break;
				case COND_GE:
					if (!((!!(S_REG[PSW]&PSW_S))^(!!(S_REG[PSW]&PSW_OV)))) P_REG[arg2] = 1;
					break;
				case COND_GT:
					if (!(((!!(S_REG[PSW]&PSW_S))^(!!(S_REG[PSW]&PSW_OV)))|(S_REG[PSW]&PSW_Z))) P_REG[arg2] = 1;
					break;
			}
			break;

          case CMP_I:
            flags = 0;
			temp = (INT64)((INT64U)(P_REG[arg2])-(INT64U)(sign_5(arg1)));

            if ((long)temp == 0) flags = flags | PSW_Z;
            if ((long)temp & 0x80000000)  flags = flags | PSW_S;
            if(((P_REG[arg2]^(sign_5(arg1)))&(P_REG[arg2]^temp))&0x80000000) flags = flags | PSW_OV;
			if ((INT64U)(temp) >> 32) flags = flags | PSW_CY;

            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            break;

          case SHL_I:
            flags = 0;
            if((arg1)&&(P_REG[arg2] >> (32 - arg1))&0x01) flags = flags | PSW_CY;
            // set CY before we destroy the regisrer info....
            P_REG[arg2] = P_REG[arg2] << arg1;
            // Set Flags
            if (P_REG[arg2] == 0) flags = flags | PSW_Z;
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            break;

          case SHR_I:
            flags = 0;
            if((arg1)&&(P_REG[arg2] >> (arg1-1))&0x01) flags = flags | PSW_CY;
            // set CY before we destroy the regisrer info....
            P_REG[arg2] = P_REG[arg2] >> arg1;
            // Set Flags
            if (P_REG[arg2] == 0) flags = flags | PSW_Z;
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            break;

		  case CLI:
            S_REG[PSW] = S_REG[PSW] & (0xFFFFEFFF);
            break;

          case SAR_I:
            flags = 0;
            msb = P_REG[arg2] & 0x80000000; // Grab the MSB

			if( (arg1) && ((P_REG[arg2]>>(arg1-1))&0x01) )
				flags = flags | PSW_CY;

            for(j = 0; j < arg1; j++)
				P_REG[arg2] = (P_REG[arg2] >> 1) | msb; //Keep sticking the msb on the end

            // Set Flags
            if (P_REG[arg2] & 0x80000000)  flags = flags | PSW_S;
            if (!P_REG[arg2]) flags = flags | PSW_Z;
            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            break;

          case TRAP:
            dtprintf(6,ferr,"\nUnhandled opcode! trap");
            break;

          case LDSR:
            S_REG[(arg1 & 0x1F)] = P_REG[(arg2 & 0x1F)];
            break;

          case STSR:
            P_REG[(arg2 & 0x1F)] = S_REG[(arg1 & 0x1F)];
            break;

          case SEI:
            S_REG[PSW] = S_REG[PSW] | 0x00001000;
            break;

          case BV:
            if(S_REG[PSW]&PSW_OV) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BL:
            if(S_REG[PSW]&PSW_CY) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BE:
            if(S_REG[PSW]&PSW_Z) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BNH:
            if((S_REG[PSW]&PSW_Z)||(S_REG[PSW]&PSW_CY)) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BN:
            if(S_REG[PSW]&PSW_S) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BR:
            PC += (sign_9(arg1) & 0xFFFFFFFE);
			clocks += 2;
            break;
          case BLT:
            if((!!(S_REG[PSW]&PSW_S))^(!!(S_REG[PSW]&PSW_OV))) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BLE:
            if(((!!(S_REG[PSW]&PSW_S))^(!!(S_REG[PSW]&PSW_OV)))||(S_REG[PSW]&PSW_Z)) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BNV:
            if(!(S_REG[PSW]&PSW_OV)) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BNL:
            if(!(S_REG[PSW]&PSW_CY)) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BNE:
            if((S_REG[PSW]&PSW_Z) == PSW_Z) {
                PC +=2;
            } else {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            }
            break;

          case BH:
            if(!((S_REG[PSW]&PSW_Z)||(S_REG[PSW]&PSW_CY))) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BP:
            if(!(S_REG[PSW] & PSW_S)) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case NOP:
            //Its a NOP do nothing =)
            PC +=2;
            break;

          case BGE:
            if(!((!!(S_REG[PSW]&PSW_S))^(!!(S_REG[PSW]&PSW_OV)))) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
            }
            break;

          case BGT:
            if(!(((!!(S_REG[PSW]&PSW_S))^(!!(S_REG[PSW]&PSW_OV)))||(S_REG[PSW]&PSW_Z))) {
                PC += (sign_9(arg1) & 0xFFFFFFFE);
				clocks += 2;
            } else {
                PC +=2;
			}
			break;

          case JR:
            PC += (sign_26(arg1) & 0xFFFFFFFE);
            break;

          case JAL:
            P_REG[31]=PC+4;
            PC += (sign_26(arg1) & 0xFFFFFFFE);
            break;

          case MOVEA:
            P_REG[arg3] = P_REG[arg2] + sign_16(arg1);
            break;

          case ADDI:
            flags = 0;
            temp = P_REG[arg2] + sign_16(arg1);
            // Set Flags
            if ((long)temp == 0) flags = flags | PSW_Z;
            if ((long)temp & 0x80000000)  flags = flags | PSW_S;
            if (((P_REG[arg2]^(~sign_16(arg1)))&(P_REG[arg2]^temp))&0x80000000) flags = flags | PSW_OV;
			if (temp < P_REG[arg2]) flags = flags | PSW_CY;

            S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
            P_REG[arg3] = (long)temp;
            break;

          case ORI:
            flags = 0;
            P_REG[arg3] = arg1 | P_REG[arg2];
            // Set Flags
            if (P_REG[arg3] == 0) flags = flags | PSW_Z;
            if (P_REG[arg3] & 0x80000000)  flags = flags | PSW_S;
            S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;
            break;

          case ANDI:
            flags = 0;
            P_REG[arg3] = (arg1 & P_REG[arg2]);
            // Set Flags
            if (P_REG[arg3] == 0) flags = (flags | PSW_Z);
            S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;
            break;

          case XORI:
			flags = 0;
			P_REG[arg3] = arg1 ^ P_REG[arg2];
			// Set Flags
			if (P_REG[arg3] == 0) flags = flags | PSW_Z;
			if (P_REG[arg3] & 0x80000000)  flags = flags | PSW_S;
			S_REG[PSW] = (S_REG[PSW] & (0xFFFFFFF0|PSW_CY))|flags;
			break;

          case MOVHI:
            P_REG[arg3] = (arg1 << 16) + P_REG[arg2];
			break;

          case RETI:
            //Return from Trap/Interupt
            if(S_REG[PSW] & PSW_NP) { // Read the FE Reg
                PC = S_REG[FEPC];
                S_REG[PSW] = S_REG[FEPSW];
            } else { 	//Read the EI Reg Interupt
                PC = S_REG[EIPC];
                S_REG[PSW] = S_REG[EIPSW];
            }
            break;

          case HALT:
            dtprintf(6,ferr,"\nUnhandled opcode! halt");
            break;

          case LD_B:
			tmp2 = (sign_16(arg1)+P_REG[arg2])&0x07FFFFFF;
			
#ifndef DBG_PRINT
			//avoid calling rbyte for most frequent cases
			if ((tmp2 & 0x7000000)==0x5000000) {
				P_REG[arg3] = sign_8(((BYTE *)(V810_VB_RAM.off + (tmp2 & V810_VB_RAM.highaddr)))[0]);
			} else if ((tmp2 & 0x7000000)==0x7000000) {
				P_REG[arg3] = sign_8(((BYTE *)(V810_ROM1.off + (tmp2 & V810_ROM1.highaddr)))[0]);
			} else 
#endif
				P_REG[arg3] = sign_8(mem_rbyte(tmp2));
			
			//should be 3 clocks when executed alone, 2 when precedes another LD, or 1
			//when precedes an instruction with many clocks (I'm guessing FP, MUL, DIV, etc)
			if (lastclock < 6) {
				if ((lastop == LD_B) || (lastop == LD_H) || (lastop == LD_W)) clocks += 1;
				else clocks += 2;
			}
            break;

          case LD_H:
			tmp2 = (sign_16(arg1)+P_REG[arg2]) & 0x07FFFFFE;

#ifndef DBG_PRINT
			//avoid calling rhword for most frequent cases
			if((tmp2 & 0x7000000) == 0x5000000) {
				P_REG[arg3] = sign_16(((HWORD *)(V810_VB_RAM.off + (tmp2 & V810_VB_RAM.highaddr)))[0]);
			} else if((tmp2 & 0x7000000) == 0x7000000) {
        		P_REG[arg3] = sign_16(((HWORD *)(V810_ROM1.off + (tmp2 & V810_ROM1.highaddr)))[0]);
			} else if((tmp2 >= V810_VIPCREG.lowaddr)&&(tmp2 <=V810_VIPCREG.highaddr)) {
				P_REG[arg3] = sign_16(((HWORD *)(&tVIPREG))[(tmp2&0x7E)>>1]);
			} else 
#endif
	            P_REG[arg3] = sign_16(mem_rhword(tmp2));

            break;

          case LD_W:
			tmp2 = (sign_16(arg1)+P_REG[arg2]) & 0x07FFFFFC;

#ifndef DBG_PRINT
			//avoid calling rword for most frequent cases
			if((tmp2 & 0x7000000) == 0x5000000) {
				P_REG[arg3] = ((WORD *)(V810_VB_RAM.off + (tmp2 & V810_VB_RAM.highaddr)))[0];
			} else if((tmp2 & 0x7000000) == 0x7000000) {
        		P_REG[arg3] = ((WORD *)(V810_ROM1.off + (tmp2 & V810_ROM1.highaddr)))[0];
			} else 
#endif
				P_REG[arg3] = mem_rword(tmp2);
			

			if (lastclock < 6) {
				if ((lastop == LD_B) || (lastop == LD_H) || (lastop == LD_W)) clocks += 3;
				else clocks += 4;
			}
            break;

          case ST_B:
            mem_wbyte(sign_16(arg2)+P_REG[arg3],P_REG[arg1]&0xFF);
			//clocks should be 2 clocks when follows another ST
			if (lastop == ST_B) clocks += 1;
            break;

          case ST_H:
            mem_whword((sign_16(arg2)+P_REG[arg3])&0xFFFFFFFE,P_REG[arg1]&0xFFFF);
			if (lastop == ST_H) clocks += 1;
            break;

          case ST_W:
			tmp2 = (sign_16(arg2)+P_REG[arg3]) & 0x07FFFFFC;

#ifndef DBG_PRINT
			//avoid calling wword for most frequent cases
			if((tmp2 & 0x7000000) == 0x5000000) {
		        ((WORD *)(V810_VB_RAM.off + (tmp2 & V810_VB_RAM.highaddr)))[0] = P_REG[arg1];
			} else 
#endif
	            mem_wword(tmp2,P_REG[arg1]);

			if (lastop == ST_W) clocks += 3;
            break;

          case IN_B:
            P_REG[arg3] = port_rbyte(sign_16(arg1)+P_REG[arg2]);
            break;

          case IN_H:
            P_REG[arg3] = port_rhword((sign_16(arg1)+P_REG[arg2]) & 0xFFFFFFFE);
            break;

          case CAXI:
            dtprintf(6,ferr,"\nUnhandled opcode! caxi");
            break;

          case IN_W:
            P_REG[arg3] = port_rword((sign_16(arg1)+P_REG[arg2]) & 0xFFFFFFFC);
            break;

          case OUT_B:
            port_wbyte(sign_16(arg2)+P_REG[arg3],P_REG[arg1]&0xFF);
			//clocks should be 2 when follows another OUT
			if (lastop == OUT_B) clocks += 1;
            break;

          case OUT_H:
            port_whword((sign_16(arg2)+P_REG[arg3])&0xFFFFFFFE,P_REG[arg1]&0xFFFF);
			if (lastop == OUT_H) clocks += 1;
            break;

          case OUT_W:
            port_wword((sign_16(arg2)+P_REG[arg3])&0xFFFFFFFC,P_REG[arg1]);
			if (lastop == OUT_W) clocks += 3;
            break;

          case FPP:
			fpu_subop(arg3,arg1,arg2);
            break;

          case BSTR:
            bstr_subop(arg2,arg1);    
            break;

          default:
            dtprintf(6,ferr,"\n%08lx\t\t%2x %2x  ;Invalid Opcode", PC, lowB, highB);
            break;
        }

		if(dbg_trc)
			return 1;
    }
}


//Bitstring routines, wrapper functions for bitstring instructions!
void get_bitstr(WORD *str, WORD src, WORD srcoff, WORD len) {
	WORD i=0,tword,tmp;

if(srcoff!=0)
	i=i;

	memset(str,0,(((len>>5)+1)<<2)); //clear bitstring data ((len/32)+1)*4

	tmp = ((i+srcoff)>>5);
	tword = mem_rword(src+(tmp<<2));
	while (i < len) {
		//if next byte, grab it
		if (((i+srcoff)>>5) != tmp) {
			tmp = ((i+srcoff)>>5);
			tword = mem_rword(src+(tmp<<2));
		}
		str[i>>5] |= (((tword >> ((srcoff+i)&0x1F)) & 1) << (i&0x1F));
		i++;
	}
}

void set_bitstr(WORD *str, WORD dst, WORD dstoff, WORD len) {
	WORD i=0,tword,tmp;

if(dstoff!=0)
	i=i;

	tmp = ((i+dstoff)>>5);
	tword = mem_rword(dst+(tmp<<2));
	while (i < len) {
		if (((i+dstoff)>>5) != tmp) {
			tmp = ((i+dstoff)>>5);
			tword = mem_rword(dst+(tmp<<2));
		}
		tword &= (~(1<<((dstoff+i)&0x1F)));
		tword |= (((str[i>>5]>>(i&0x1F))&1)<<((dstoff+i)&0x1F));
		i++;
		if (!((i+dstoff)&0x1F)) mem_wword(dst+(tmp<<2),tword);
	}
	mem_wword(dst+(tmp<<2),tword);
}


int bstr_subop(int sub_op, int arg1) {
	WORD i,tmp[8192],tmp2[8192];

	WORD dstoff = (P_REG[26] & 0x1F);
	WORD srcoff = (P_REG[27] & 0x1F);
	WORD len =     P_REG[28];
	WORD dst =    (P_REG[29] & 0xFFFFFFFC);
	WORD src =    (P_REG[30] & 0xFFFFFFFC);


    if(sub_op > 15) {
        dtprintf(10,ferr,"\n%08lx\tBSR Error: %04x", PC,sub_op);
		return 0;
    }

    switch(sub_op) {
	case SCH0BSU:
		dtprintf(10,ferr,"\nSCH0BSU, len: %08X, src: %08X, srcoff: %08X, dst: %08X, dstoff: %08X",len,src,srcoff,dst,dstoff);
		break;

	case SCH0BSD:
		dtprintf(10,ferr,"\nSCH0BSD, len: %08X, src: %08X, srcoff: %08X, dst: %08X, dstoff: %08X",len,src,srcoff,dst,dstoff);
		break;

	case SCH1BSU:
		dtprintf(10,ferr,"\nSCH1BSU, len: %08X, src: %08X, srcoff: %08X, dst: %08X, dstoff: %08X",len,src,srcoff,dst,dstoff);
		break;

	case SCH1BSD:
		dtprintf(10,ferr,"\nSCH1BSD, len: %08X, src: %08X, srcoff: %08X, dst: %08X, dstoff: %08X",len,src,srcoff,dst,dstoff);
		break;

	case ORBSU:
		get_bitstr(tmp,src,srcoff,len);
		get_bitstr(tmp2,dst,dstoff,len);
		for (i = 0; i < ((len>>5)+1); i++) tmp[i] |= tmp2[i];
		set_bitstr(tmp,dst,dstoff,len);
		break;

	case ANDBSU:
		get_bitstr(tmp,src,srcoff,len);
		get_bitstr(tmp2,dst,dstoff,len);
		for (i = 0; i < ((len>>5)+1); i++) tmp[i] &= tmp2[i];
		set_bitstr(tmp,dst,dstoff,len);
		break;

	case XORBSU:
		get_bitstr(tmp,src,srcoff,len);
		get_bitstr(tmp2,dst,dstoff,len);
		for (i = 0; i < ((len>>5)+1); i++) tmp[i] ^= tmp2[i];
		set_bitstr(tmp,dst,dstoff,len);
		break;

	case MOVBSU:
		get_bitstr(tmp,src,srcoff,len);
		set_bitstr(tmp,dst,dstoff,len);
		break;

	case ORNBSU:
		get_bitstr(tmp,src,srcoff,len);
		get_bitstr(tmp2,dst,dstoff,len);
		for (i = 0; i < ((len>>5)+1); i++) tmp[i] = (~tmp[i] | tmp2[i]);
		set_bitstr(tmp,dst,dstoff,len);
		break;

	case ANDNBSU:
		get_bitstr(tmp,src,srcoff,len);
		get_bitstr(tmp2,dst,dstoff,len);
		for (i = 0; i < ((len>>5)+1); i++) tmp[i] = (~tmp[i] & tmp2[i]);
		set_bitstr(tmp,dst,dstoff,len);
		break;

	case XORNBSU:
		get_bitstr(tmp,src,srcoff,len);
		get_bitstr(tmp2,dst,dstoff,len);
		for (i = 0; i < ((len>>5)+1); i++) tmp[i] = (~tmp[i] ^ tmp2[i]);
		set_bitstr(tmp,dst,dstoff,len);
		break;

	case NOTBSU:
		get_bitstr(tmp,src,srcoff,len);
		for (i = 0; i < ((len>>5)+1); i++) tmp[i] = ~tmp[i];
		set_bitstr(tmp,dst,dstoff,len);
		break;

	default:
        dtprintf(10,ferr,"\n%08lx\tBSR Error: %04x", PC,sub_op);
		break;
	}

	return 0;
}

int fpu_subop(int sub_op, int arg1, int arg2) {
	int i, flags = 0; // Set Flags, OV set to Zero
	double dTemp;
	float fTemp;
	int temp;

    if(sub_op > 15) {
        dtprintf(10,ferr,"\n%08lx\tFPU Error: %04x", PC, sub_op);
		return 0;
    }

	switch(sub_op) {
	case CMPF_S:
		dTemp = (double)(*((float *)&P_REG[arg1])) - (double)(*((float *)&P_REG[arg2]));

		if (dTemp == 0.0F) flags = flags | PSW_Z;
		if (dTemp < 0.0F)  flags = flags | PSW_S;
		if (dTemp > ((float)dTemp)) flags = flags | PSW_CY; //How???
		S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;
		break;

	case CVT_WS:
		fTemp = (float)((long)P_REG[arg2]);

		if (fTemp == 0) flags = flags | PSW_Z;
		if (fTemp < 0.0F)  flags = flags | PSW_S;
		if (P_REG[arg2] != fTemp) flags = flags | PSW_CY; //How???
		S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;

		P_REG[arg1] = *((WORD *)&fTemp);
		break;

	case CVT_SW:
		P_REG[arg1] = (long)(*((float *)&P_REG[arg2])+0.5F);

		if (P_REG[arg1] == 0) flags = flags | PSW_Z;
		if (P_REG[arg1] & 0x80000000)  flags = flags | PSW_S;
		S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF7)|flags;
		break;

	case ADDF_S:
		dTemp = (double)(*((float *)&P_REG[arg1])) + (double)(*((float *)&P_REG[arg2]));

		if (dTemp == 0.0F) flags = flags | (PSW_Z | PSW_CY);  //changed by frostgiant based on NEC docs
		if (dTemp < 0.0F)  flags = flags | PSW_S;

		S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;

		fTemp = ((float)dTemp);
		P_REG[arg1] = *((WORD *)&fTemp);
		break;

	case SUBF_S:
		dTemp = (double)(*((float *)&P_REG[arg1])) - (double)(*((float *)&P_REG[arg2]));

		if (dTemp == 0.0F) flags = flags | (PSW_Z | PSW_CY);  //changed by frostgiant based on NEC docs
		if (dTemp < 0.0F)  flags = flags | PSW_S;

		S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;

		fTemp = ((float)dTemp);
		P_REG[arg1] = *((WORD *)&fTemp);
		break;

	case MULF_S:
		dTemp = (double)(*((float *)&P_REG[arg1])) * (double)(*((float *)&P_REG[arg2]));

		if (dTemp == 0.0F) flags = flags | (PSW_Z | PSW_CY);  //changed by frostgiant based on NEC docs
		if (dTemp < 0.0F)  flags = flags | PSW_S;
		S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;

		fTemp = ((float)dTemp);
		P_REG[arg1] = *((WORD *)&fTemp);
		break;

	case DIVF_S:
		dTemp = (double)(*((float *)&P_REG[arg1])) / (double)(*((float *)&P_REG[arg2]));

		if (dTemp == 0.0F) flags = flags | (PSW_Z | PSW_CY);  //changed by frostgiant based on NEC docs
		if (dTemp < 0.0F)  flags = flags | PSW_S;

		S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF0)|flags;

		fTemp = ((float)dTemp);
		P_REG[arg1] = *((WORD *)&fTemp);
		break;

	case XB:
		if (arg2) dtprintf(10,ferr,"\nXB Instruction, arg2 = r%d",arg2);
		P_REG[arg1] = ((P_REG[arg1]&0xFFFF0000) | (((P_REG[arg1]<<8)&0xFF00) | ((P_REG[arg1]>>8)&0xFF)));
		break;

	case XH:
		if (arg2) dtprintf(10,ferr,"\nXH Instruction, arg2 = r%d",arg2);
		P_REG[arg1] = (P_REG[arg1]<<16)|(P_REG[arg1]>>16);
		break;

	case REV:
		temp = 0;
		for (i = 0; i < 32; i++) temp = ((temp << 1) | ((P_REG[arg2] >> i) & 1));
		P_REG[arg1] = temp;
		break;

	case TRNC_SW:
		P_REG[arg1] = (WORD)(*((float *)&P_REG[arg2])+0.5F);

		if (!P_REG[arg1]) flags = flags | PSW_Z;
		if (P_REG[arg1] & 0x80000000)  flags = flags | PSW_S;
		S_REG[PSW] = (S_REG[PSW] & 0xFFFFFFF7)|flags;
		break;

	case MPYHW:
		//if (P_REG[arg1] & 0xFFFF0000) dtprintf(10,ferr,"\nMPYHW Instruction, arg1 = %08X",P_REG[arg1]);
		//if (P_REG[arg2] & 0xFFFF0000) dtprintf(10,ferr,"\nMPYHW Instruction, arg2 = %08X",P_REG[arg2]);
		P_REG[arg1 & 0x1F] = P_REG[arg1 & 0x1F] * P_REG[arg2 & 0x1F];
		break;

	default:
        dtprintf(10,ferr,"\n%08lx\tFPU Error: %04x", PC, sub_op);
		break;
	}

	return 0;
}


VB_INLINE void serviceInt(unsigned int cycles) {
	static unsigned int lasttime=0;

	//OK, this is a strange muck of code... basically it attempts to hit interrupts and
	//handle the VIP regs at the correct time. The timing needs a LOT of work. Right now,
	//the count values I'm using are the best values from my old clock cycle table. In
	//other words, the values are so far off. PBBT!  FIXME

	//For whatever reason we dont need this code
	//actualy it totaly breaks the emu if you don't call it on
	//every cycle, fixme, what causes this to error out.
	//Controller Int
	//if ((!(tHReg.SCR & 0x80)) && (handle_input()&0xFFFC)) {
	//	v810_int(0);
	//}

	if (tHReg.TCR & 0x01) { // Timer Enabled
		if ((cycles-lasttime) > tHReg.tTRC) {
			if (tHReg.tCount)
				tHReg.tCount--;
			tHReg.TLB = (tHReg.tCount&0xFF);
			tHReg.THB = ((tHReg.tCount>>8)&0xFF);
			lasttime=cycles;
			if (tHReg.tCount == 0) {
				tHReg.tCount = tHReg.tTHW; //reset counter
				tHReg.TCR |= 0x02; //Zero Status
				if (tHReg.TCR & 0x08) {
					v810_int(1);
				}
			}
		}
	}

}


VB_INLINE int serviceDisplayInt(unsigned int cycles) {
	static unsigned int lastfb=0;
	static int rowcount,tmp1,frames=0;
	int gamestart;
	unsigned int tfb = (cycles-lastfb);


	//Handle DPSTTS, XPSTTS, and Frame interrupts
	if (rowcount < 0x1C) {
		if ((rowcount == 0) && (tfb > 0x0210) && (!tmp1)) {
			tmp1=1;
			tVIPREG.XPSTTS &= 0x000F;
			tVIPREG.DPSTTS = ((tVIPREG.DPCTRL&0x0302)|0xC0);
			if (++frames > tVIPREG.FRMCYC) {
				frames = 0;
				gamestart = 0x0008;
			} else {
				gamestart = 0;
			}
			if (tVIPREG.INTENB&(0x0010|gamestart))
				v810_int(4);
			tVIPREG.INTPND |= (0x0010|gamestart);
			return 1;
		}
		if ((tfb > 0x0500) && (!(tVIPREG.XPSTTS&0x8000))) 
			tVIPREG.XPSTTS |= 0x8000;
		if (tfb > 0x0A00) {
			tVIPREG.XPSTTS = ((tVIPREG.XPSTTS&0xE0)|(rowcount<<8)|(tVIPREG.XPCTRL & 0x02));
			rowcount++;
			lastfb=cycles;
		}
		if ((rowcount == 0x12) && (tfb > 0x670))
			tVIPREG.DPSTTS = ((tVIPREG.DPCTRL&0x0302)|(tVIPREG.tFrame&1?0xD0:0xC4));
	} else {
		if ((rowcount == 0x1C) && (tfb > 0x10000)) {			//0x100000
			tVIPREG.XPSTTS = (0x1B00|(tVIPREG.XPCTRL & 0x02));

			if(tVBOpt.VFHACK)					//vertical force hack
				v810_int(4);
			else if (tVIPREG.INTENB&0x4000) 
				v810_int(4);					//XPEND

			tVIPREG.INTPND |= 0x4000;				//(tVIPREG.INTENB&0x4000);
			rowcount++;
		} else if ((rowcount == 0x1D) && (tfb > 0x18000)) {		//0xE690
			tVIPREG.DPSTTS = ((tVIPREG.DPCTRL&0x0302)|0xC0);
			if (tVIPREG.INTENB&0x0002) 
				v810_int(4);					//LFBEND
			tVIPREG.INTPND |= 0x0002;				//(tVIPREG.INTENB&0x0002);
			rowcount++;
		} else if ((rowcount == 0x1E) && (tfb > 0x20000)) {		//0x15E70
			tVIPREG.DPSTTS = ((tVIPREG.DPCTRL&0x0302)|0x40);
			if (tVIPREG.INTENB&0x0004) 
				v810_int(4);					//RFBEND
			tVIPREG.INTPND |= 0x0004;				//(tVIPREG.INTENB&0x0004);
			rowcount++;
		} else if ((rowcount == 0x1F) && (tfb > 0x28000)) {		//0x1FAD8
			tVIPREG.DPSTTS = ((tVIPREG.DPCTRL&0x0302)|(tVIPREG.tFrame&1?0x48:0x60));
			if (tVIPREG.INTENB&0x2000) 
				v810_int(4);					//SBHIT
			tVIPREG.INTPND |= 0x2000;
			rowcount++;
		} else if ((rowcount == 0x20) && (tfb > 0x38000)) {		//0x33FD8
			tVIPREG.DPSTTS = ((tVIPREG.DPCTRL&0x0302)|0x40);
			rowcount++;
		} else if ((rowcount == 0x21) && (tfb > 0x42000)) {
			tmp1=0;
			rowcount=0;
			tVIPREG.tFrame++;
			if ((tVIPREG.tFrame < 1) || (tVIPREG.tFrame > 2)) tVIPREG.tFrame = 1;
			tVIPREG.XPSTTS = (0x1B00|(tVIPREG.tFrame<<2)|(tVIPREG.XPCTRL & 0x02));
			lastfb=cycles;
		}
	}
	return 0;
}



// Generate Interupt #n
void v810_int(WORD iNum) {
    dtprintf(4,ferr,"\nInt atempt %x",iNum);

    if (iNum > 0x0F) return;  // Invalid Interupt number...
    if((S_REG[PSW] & PSW_NP)) return;
    if((S_REG[PSW] & PSW_EP)) return; // Exception pending?
    if((S_REG[PSW] & PSW_ID)) return; // Interupt disabled
    if(iNum < ((S_REG[PSW] & PSW_IA)>>16)) return; // Interupt to low on the chain

    dtprintf(6,ferr,"\nInt %x",iNum);

    //Ready to Generate the Interupts
    S_REG[EIPC]  = PC;
    S_REG[EIPSW] = S_REG[PSW];

    PC = 0xFFFFFE00 | (iNum << 4);
    
    S_REG[ECR] = 0xFE00 | (iNum << 4);
    S_REG[PSW] = S_REG[PSW] | PSW_EP;
    S_REG[PSW] = S_REG[PSW] | PSW_ID;
    if((iNum+=1) > 0x0F) 
		(iNum = 0x0F);
    S_REG[PSW] = S_REG[PSW] | (iNum << 16); //Set the Interupt

}


// Generate exception #n
//Exceptions are Div by zero, trap and Invalid Opcode, we can live without...
void v810_exp(WORD iNum, WORD eCode) {
    if (iNum > 0x0F) return;  // Invalid Exception number...

    //if(!S_REG[PSW]&PSW_ID) return;
    //if(iNum < ((S_REG[PSW] & PSW_IA)>>16)) return; // Interupt to low on the mask level....
    if ((S_REG[PSW] & PSW_IA)>>16) return; //Interrupt Pending

	eCode &= 0xFFFF;
/*
    if(S_REG[PSW]&PSW_NP) { //Fatal Exception
        S_REG[DPC] = PC;
        S_REG[DPSW] = S_REG[PSW];
        S_REG[PSW] = S_REG[PSW] | PSW_DP;
        S_REG[PSW] = S_REG[PSW] | PSW_NP;
        S_REG[PSW] = S_REG[PSW] | PSW_EP;
        S_REG[PSW] = S_REG[PSW] | PSW_ID;
        //S_REG[PSW] = S_REG[PSW] | (((iNum+1) & 0x0f) << 16); //Set the Interupt status

        PC = 0xFFFFFFE0;
        return;
    }else
*/
    if(S_REG[PSW]&PSW_EP) { //Double Exception
        S_REG[FEPC] = PC;
        S_REG[FEPSW] = S_REG[PSW];
        S_REG[ECR] = (eCode << 16); //Exception Code, dont get it???
        S_REG[PSW] = S_REG[PSW] | PSW_NP;
        S_REG[PSW] = S_REG[PSW] | PSW_ID;
        //S_REG[PSW] = S_REG[PSW] | (((iNum+1) & 0x0f) << 16); //Set the Interupt status

        PC = 0xFFFFFFD0;
        return;
    } else {                                // Regular Exception
        S_REG[EIPC] = PC;
        S_REG[EIPSW] = S_REG[PSW];
        S_REG[ECR] = eCode; //Exception Code, dont get it???
        S_REG[PSW] = S_REG[PSW] | PSW_EP;
        S_REG[PSW] = S_REG[PSW] | PSW_ID;
        //S_REG[PSW] = S_REG[PSW] | (((iNum+1) & 0x0f) << 16); //Set the Interupt status

        PC = 0xFFFFFF00 | (iNum << 4);
        return;
    }
}



